/******************************************************************************
*                                                                             *
* Freescale Semiconductor Inc.                                                *
* (c) Copyright 2004-2008 Freescale Semiconductor, Inc.                       *
* ALL RIGHTS RESERVED.                                                        *
*                                                                             *
*******************************************************************************
*******************************************************************************
* Example for S08QE8  - I2C Slave + IR decoder to GPIO port                   *
*                       I2C max speed 100kHz;                                 *
*                       IR - RC-5 frame;                                      *
******************************************************************************/

#include <hidef.h>          /* for EnableInterrupts macro */
#include "derivative.h"     /* include peripheral declarations */
#include "application.h"    /* difinitions & declarations */

//*****************************************************************************
void main(void) 
{
    MCU_init(); /* call Device Initialization */
    Init_Variables();
    
    PTAD_PTAD1 = 1;                 // Button as input;
    PTADD = 0;
    PTBD_PTBD0 = 1;                 // IR Receiver input;
    PTBDS = 0x80;
    PTBDD = 0x80;                   // PTB7 as output;
    
    for(;;) 
    {
        __RESET_WATCHDOG();
        
        if(mode == SLEEP)
        {
            KBIPE = 0x12;           // set KBI input pins;
            KBISC_KBACK = 1;        // Clear the KBI flag;
            KBISC_KBIE = 1;         // Enable KBI interrupt;
            __RESET_WATCHDOG();
            asm stop;
        }

        asm nop;
        
        switch(flag)
        {
            case STBY:
            {
                STBY_Button();
                flag = 0;
                KBIPE = 0x12;       // Enable KBI;
                KBISC_KBACK = 1;    // Clear the KBI flag;
                KBISC_KBIE = 1;     // Enable KBI interrupt;
            }
            break;
            
            case RC5:
            {
                IR_receive();
                Compare_IR();
                KBIPE = 0x12;       // Enable KBI;
                KBISC_KBACK = 1;    // Clear the KBI flag;
                KBISC_KBIE = 1;     // Enable KBI interrupt;
                flag = 0;
            }
            break;
            
            case I2C:
            {
                Compare_I2C();
                flag = 0;
            }
            break;
            
            default:
            break;
        }
        
        
        
        if(address_valid && command_ON && (PWR_PIN == OFF)) 
        {
            PWR_PIN = ON;           // switch ON the main power supply;
            mode = RUN;             // set mode of the SPMU;
        }
        if(address_valid && command_OFF && (PWR_PIN == ON)) 
        {
            PWR_PIN = OFF;          // switch OFF the main power supply;
            mode = SLEEP;           // set mode of the SPMU;
        }
        
        address_valid = FALSE;
        command_ON = FALSE;
        command_OFF = FALSE;
    }
}
/********************************************************************************
*   Description: Keyboard interrupt service routine;                            *
*                                                                               *
*   Input:  PTB5 pin;                                                           *
*                                                                               *
*   Output: flag = 1;                                                           *
*                                                                               *
********************************************************************************/ 
__interrupt IV(Vkeyboard) void isrVkeyboard(void)
{
    KBISC = 0;
    KBISC_KBIE = 0;                 // disable KBI interrupts;
    KBISC_KBACK = 1;                // clear KBI flag;
    if(!RC5_PIN) flag = RC5;
    else if(!BUTTON) flag = STBY;
}
/********************************************************************************
*   Description: IIC interrupt service routine;                                 *
*                                                                               *
*   Input:  SDA, SCL pins;                                                      *
*                                                                               *
*   Output: RX_data byte;                                                       *
*                                                                               *
********************************************************************************/ 
__interrupt IV(Viic) void isrViic(void)
{
    if(IICS_ARBL) IICS_ARBL = 1;    // Clear the ARBL flag;
    if(!IICS_IAAS) 
    {
        IICC_IICEN = 1;
        IICS = IICS_IICIF_MASK;     // clear the flag;
        return;
    }
    else
    {
        if(i2c_cnt == 0)
        {
            Address = IICD;         // save received Address;
        } 
        else if(i2c_cnt == 1)
        {
            Command = IICD;         // save received Command;
        }
    }
    IICS = IICS_IICIF_MASK;         // clear the flag;
    i2c_cnt++;
    if(i2c_cnt > 1)
    {
        flag = I2C;
        i2c_cnt = 0;
    }
}

/********************************************************************************
*   Description: Compares the "Address" and "Command" bytes recsived by I2C;    *
*                                                                               *
*   Input:  - "Address" and "Command" bytes;                                    *
*                                                                               *
*   Output: - STB power supply switch OFF or other function;                    *
*                                                                               *
********************************************************************************/
void Compare_I2C(void)
{
    address_valid = FALSE;
    switch (Address)
    {
        case MY_I2C_ADDR:
        {
            address_valid = TRUE;
        } 
        break;
    }
    
    command_OFF = FALSE;
    switch (Command)
    {
        case CMD_OFF1:
        {
            command_OFF = TRUE;
        } 
        break;
    }
}
/********************************************************************************
*   Time delay functions                                                        *
********************************************************************************/
void Delay(word del)                // Universal delay;
{
    word i;
    i = del;
    while(i > 0)
    {
        i--;
        __RESET_WATCHDOG();
    }
}
//-----------------------------------------------------------------------------
void Wait1ms(void)                  // for 8,388MHz BusCLK;
{
    Delay(750);
}
//-----------------------------------------------------------------------------
void WaitNms(byte del)              // for 8,388MHz BusCLK;
{
    byte i;
    i = del;
    while(i > 0) 
    {
        Wait1ms();
        i--;
    }
}
/********************************************************************************
*   Description: Receives RC-5 code;                                            *
*                                                                               *
*   Input:  IR RC-5 demodulated signal on RC5_PIN;                              *
*                                                                               *
*   Output: "Address" and "Command" bytes;                                      *
*                                                                               *
********************************************************************************/
void IR_receive(void)                   // for Bus = 8.388MHz;
{
    byte i;
    KBISC_KBIE = 0;                    // Disable KBI interrupt;
    KBISC_KBACK = 1;                   // Clear the flag;
    IR_data = 0;
    //;--------- receive first bit(S0) ---------------------
    Delay(128);
    if(!RC5_PIN) IR_data |= 0x01;       // read RC5 = 1;
    IR_data <<= 1;
    //;--------- receive (S1,T0) + Address (A4 - A0) -------
    for(i=7;i>0;i--)
    {
        Delay(490);
        if(!RC5_PIN) IR_data |= 0x01;   // read RC5 = 1;
        IR_data <<= 1;
    }
    Address = (IR_data >> 1) & 0x1F;
    IR_data = 0;
    //;--------- receive Command bits(D5 - D0) -------------
    for(i=6;i>0;i--)
    {
        Delay(490);
        if(!RC5_PIN) IR_data |= 0x01;   // read RC5 = 1;
        IR_data <<= 1;
    }
    
    Command = (IR_data >> 1) & 0x3F;
    KBISC_KBIE = 1;
}
/********************************************************************************
*   Description: Senses the user button;                                        *
*                                                                               *
*   Input:  STBY button pressed;                                                *
*                                                                               *
*   Output: variable:  mode;                                                    *
*                                                                               *
********************************************************************************/
void STBY_Button(void)  /* service the STBY button in RUN or SLEEP mode */
{
    WaitNms(50);
    
    asm nop;
    
    if (!BUTTON) while (!BUTTON) __RESET_WATCHDOG();;
    if (mode == RUN)
    {
        PWR_PIN = OFF;
        mode = SLEEP;
    }
    else if (mode == SLEEP)
    {
        PWR_PIN = ON;
        mode = RUN;
    }
    WaitNms(250);
}
/********************************************************************************
*   Description: set initial values of variebles;                               *
*                                                                               *
*   Input:  NO;                                                                 *
*                                                                               *
*   Output: default values for variables;                                       *
*                                                                               *
********************************************************************************/
void Init_Variables(void)
{
    PWR_PIN = ON;
    mode = RUN;
    command_ON = FALSE;
    command_OFF = FALSE;
    flag = 0;
    r_w = 0;
    RX_data = 0;
    TX_data = 0;
}
/********************************************************************************
*   Description: Compares received IR data;                                     *
*                                                                               *
*   Input:  variables "Address" and "Command";                                  *
*                                                                               *
*   Output: variables: - address_valid, command_ON, command_OFF;                *
*                                                                               *
********************************************************************************/
void Compare_IR(void)
{
    address_valid = FALSE;
    switch (Address)
    {
        case ADDR1:
        case ADDR2:
        case ADDR3:
        case ADDR4:
        case ADDR5:
        case ADDR6:
        case ADDR7:
        case ADDR8:
        case ADDR9:
        case ADDR10:
        case ADDR11:
        case ADDR12:
        {
            address_valid = TRUE;
        } break;
    }
    
    command_ON = FALSE;
    switch (Command)
    {
        case CMD_ON1:
        case CMD_ON2:
        case CMD_ON3:
        case CMD_ON4:
        case CMD_ON5:
        case CMD_ON6:
        case CMD_ON7:
        case CMD_ON8:
        case CMD_ON9:
        case CMD_ON10:
        case CMD_ON11:
        {
            command_ON = TRUE;
        } break;
    }
    
    command_OFF = FALSE;
    switch (Command)
    {
        case CMD_OFF1:
        {
            command_OFF = TRUE;
        } break;
    }
}
//-----------------------------------------------------------------------------

extern near void _Startup(void);

/* Interrupt vector table */
#ifndef UNASSIGNED_ISR
  #define UNASSIGNED_ISR 0xFFFF        /* unassigned interrupt service routine */
#endif

void (* near const _vect[])() @0xFFCE = { /* Interrupt vector table */
         UNASSIGNED_ISR,               /* Int.no.  0 Vrtc (at FFCE)                  Unassigned */
         UNASSIGNED_ISR,               /* Int.no.  1 VReserved23 (at FFD0)           Unassigned */
         UNASSIGNED_ISR,               /* Int.no.  2 VReserved22 (at FFD2)           Unassigned */
         UNASSIGNED_ISR,               /* Int.no.  3 VReserved21 (at FFD4)           Unassigned */
         UNASSIGNED_ISR,               /* Int.no.  4 Vacmpx (at FFD6)                Unassigned */
         UNASSIGNED_ISR,               /* Int.no.  5 Vadc (at FFD8)                  Unassigned */
         isrVkeyboard,                 /* Int.no.  6 Vkeyboard (at FFDA)             Used */
         isrViic,                      /* Int.no.  7 Viic (at FFDC)                  Used */
         UNASSIGNED_ISR,               /* Int.no.  8 Vscitx (at FFDE)                Unassigned */
         UNASSIGNED_ISR,               /* Int.no.  9 Vscirx (at FFE0)                Unassigned */
         UNASSIGNED_ISR,               /* Int.no. 10 Vscierr (at FFE2)               Unassigned */
         UNASSIGNED_ISR,               /* Int.no. 11 Vspi (at FFE4)                  Unassigned */
         UNASSIGNED_ISR,               /* Int.no. 12 VReserved12 (at FFE6)           Unassigned */
         UNASSIGNED_ISR,               /* Int.no. 13 Vtpm2ovf (at FFE8)              Unassigned */
         UNASSIGNED_ISR,               /* Int.no. 14 Vtpm2ch2 (at FFEA)              Unassigned */
         UNASSIGNED_ISR,               /* Int.no. 15 Vtpm2ch1 (at FFEC)              Unassigned */
         UNASSIGNED_ISR,               /* Int.no. 16 Vtpm2ch0 (at FFEE)              Unassigned */
         UNASSIGNED_ISR,               /* Int.no. 17 Vtpm1ovf (at FFF0)              Unassigned */
         UNASSIGNED_ISR,               /* Int.no. 18 Vtpm1ch2 (at FFF2)              Unassigned */
         UNASSIGNED_ISR,               /* Int.no. 19 Vtpm1ch1 (at FFF4)              Unassigned */
         UNASSIGNED_ISR,               /* Int.no. 20 Vtpm1ch0 (at FFF6)              Unassigned */
         UNASSIGNED_ISR,               /* Int.no. 21 Vlvd (at FFF8)                  Unassigned */
         UNASSIGNED_ISR,               /* Int.no. 22 Virq (at FFFA)                  Unassigned */
         UNASSIGNED_ISR,               /* Int.no. 23 Vswi (at FFFC)                  Unassigned */
         _Startup                      /* Int.no. 24 Vreset (at FFFE)                Reset vector */
};


/********************************************************************************
********************************************************************************/



